# $Id$

package CherryPick::LicenseManager;

use strict;
use base qw( MT::ErrorHandler );
use vars qw($DAY_SECONDS);
$DAY_SECONDS = 60 * 60 * 24;

use MT;
use MT::I18N;
use Digest::MD5 qw(md5_hex);


# Constructor.
sub new {
    my $class = shift;
    my ($params) = @_;
    my $self = $class->SUPER::new(@_);
    
    # Default members.
    $self->{trial_days} = '14';
    $self->{confirm_interval_days} = '7';
    $self->{plugindata_key} = 'license';
    $self->{l10n_class} = 'CherryPick::LicenseManager::L10N';

    $self->{setup_time_name} = 'setup_time';
    $self->{licensekey_name} = 'licensekey';
    $self->{email_name} = 'email';
    $self->{last_message_name} = 'last_message';
    $self->{confirmed_time_name} = 'confirmed_time';
    $self->{confirmation_message_name} = 'confirmation_message';
    
    # Set construction values.
    foreach my $name (keys %$params) { $self->{$name} = $params->{$name} };
    
    # Normalize expires and confirmation seconds.
    if(!$self->{trial} && $self->{trial_days}) {
        $self->{trial} = $self->{trial_days} * $DAY_SECONDS; # days to seconds.
    }
    if(!$self->{confirm_interval} && $self->{confirm_interval_days}) {
        $self->{confirm_interval} = $self->{confirm_interval_days} * $DAY_SECONDS; # days to seconds.
    }
    
    # Validation.
    foreach my $member (qw(plugin setup_time_name licensekey_name email_name last_message_name confirmed_time_name confirmation_message_name trial plugindata_key)) {
        return $class->error($self->translate('CherryPick::LicenseManager new object requires [_1] member.', $member))
            unless $self->{$member};
    }
    
    # Plugin key.
    $self->{plugin_key} ||= ($self->{plugin}->key || $self->{plugin}->name);
    
    # Current time.
    my $time = time;
    
    # Set setup time.
    unless($self->setup_time) {
        $self->setup_time($time);
        $self->save;
    }
    
    # Confirm the license.
    if($self->{confirm_interval} && $self->licensekey) {
        my $confirmed_time = $self->confirmed_time || $self->setup_time;
        if(defined($self->{confirmed_interval}) && ($time - $confirmed_time > $self->{confirmed_interval})) {
            $self->confirm_license;
        }
    }
    
    return $self;
}

# Plugin component.
sub component {
    my $self = shift;
    $self->{plugin} || MT->component('core');
}

# Translation.
sub translate {
    my $self = shift;
    my $c = $self->component;
    my $lang = MT->current_language || MT->config->DefaultLanguage;
    # Switch l10n handles and class.
    my $handles = MT->request('l10n_handle') || {};
    my $ex_handle = $handles->{$c->id};
    my $ex_class = $c->l10n_class;
    $c->l10n_class($self->{l10n_class});
    delete $handles->{$c->id};
    
    # Translate.
    my $result = $c->translate(@_);
    
    # Switch back l10n handle and class.
    delete $handles->{$c->id};
    $handles->{$c->id} = $ex_handle if $ex_handle;
    $c->l10n_class($ex_class);
    
    return $result;
}

# Get plugin data object.
sub plugin_data {
    my $self = shift;
    return $self->{plugin_data} if $self->{plugin_data};
    
    # Load plugin data.
    my $pd = MT->model('plugindata')->load({
        plugin => $self->{plugin_key},
        key => $self->{plugindata_key},
    });
    
    unless($pd) {
        # New plugin data.
        $pd = MT->model('plugindata')->new || return $self->error(MT->model('plugindata')->errstr);
        $pd->plugin($self->{plugin_key});
        $pd->key($self->{plugindata_key});
        $pd->data({})
    }
    
    $self->{plugin_data} = $pd;
}

# Save plugin data object.
sub save {
    my $self = shift;
    
    # Save plugin data.
    my $pd = $self->plugin_data || return;
    $pd->save || return $self->error($pd->errstr);
}

# Plugin data accessor.
sub data_value {
    my $self = shift;
    my ($name, $value) = @_;
    
    return '' if !$name;
    
    # Get plugin data.
    my $plugin_data = $self->plugin_data || return;
    my $data = $plugin_data->data || {};
    
    # Set a value.
    if(defined($value)) {
        $data->{$name} = $value;
        $plugin_data->data($data);
    }
    
    # Return the value.
    $data->{$name};
}

# Setup time accessor.
sub setup_time {
    my $self = shift;
    my ($time) = @_;
    
    # Use accessor.
    $self->data_value($self->{setup_time_name}, $time);
}

# Confirmation accessor.
sub confirmed_time {
    my $self = shift;
    my ($time) = @_;
    
    # Use accessor.
    $self->data_value($self->{confirmed_time}, $time);
}

# Licensekey accessor.
sub licensekey {
    my $self = shift;
    my ($licensekey) = @_;
    
    # Use accessor.
    $self->data_value($self->{licensekey_name}, $licensekey);
}

# The last message.
sub last_message {
    my $self = shift;
    my ($message) = @_;
    
    # Use accessor.
    my $value = $self->data_value($self->{last_message_name}, $message);
    
    # Save immediately.
    $self->save;

    $value;
}

# The confirmation message.
sub confirmation_message {
    my $self = shift;
    my ($message) = @_;
    
    # Use accessor.
    my $value = $self->data_value($self->{confirmation_message_name}, $message);
    
    # Save immediately.
    $self->save;
    
    $value;
}

# email.
sub email {
    my $self = shift;
    my ($email) = @_;
    
    # Use accessor.
    $self->data_value($self->{email_name}, $email);
}

# Expired flag.
sub is_expired {
    my $self = shift;
    
    my $setup_time = $self->setup_time;
    my $licensekey = $self->licensekey;
    
    return !$licensekey && time > $setup_time + $self->{trial}? 1: 0;
}

# Warning.
sub warning {
    my $self = shift;
    
    $self->confirmation_message;
}

# Remainder seconds.
sub remainder {
    my $self = shift;
    
    # Return remainder seconds.
    my $setup_time = $self->setup_time;
    my $remainder = $setup_time + $self->{trial} - time;
    
    $remainder = 0 if $remainder < 0;
    $remainder;
}

# Remainder days.
sub remainder_days {
    my $self = shift;
    
    my $remainder = $self->remainder;
    $remainder? int(($self->remainder + $DAY_SECONDS / 2) / $DAY_SECONDS): 0; #seconds to days.
}

# Registered flag.
sub is_registered {
    my $self = shift;
    my $licensekey = $self->licensekey;
    
    return $licensekey? 1: 0;
}

# Post to the server.
sub post_license {
    my $self = shift;
    my ($url, $post) = @_;
    
    # Fill the system information.
    my $app = MT->instance;
    $app = MT::App->instance unless ref($app) =~ /^MT::App/;
    $post->{pluginName} = $self->{plugin_key};
    $post->{systemLocation} = $app->base . $app->mt_uri;
    $post->{hint} ||= '';
    $post->{lang} ||= (MT->current_language || MT->config->DefaultLanguage);
    
    # LWP::UserAgent
    my $ua = MT->instance->new_ua;
    $ua->post($url, $post);
}

# Nornalize response encoding.
sub normalize_response {
    my $self = shift;
    my ($message) = shift;
    
    if(MT->version_number >= 5) {
        # Decode message if on the MT5.
        utf8::decode($message);
    } else {
        # Convert to publish charset.
        my $charset = MT->instance->config('PublishCharset') || 'UTF-8';
        $message = MT::I18N::convert_text('UTF-8', $charset, $message) unless $charset =~ /UTF-?8/i;
    }
    
    $message;
}

# Register the license.
sub register_license {
    my $self = shift;
    my ($param) = @_;
    my $post = {};
    
    # Reset the last message.
    $self->last_message('');
    
    # Check licensekey.
    unless($param->{licensekey}) {
        $self->last_message($self->translate('The license key is required.'));
        return 'error';
    }
    
    # Check email.
    unless($param->{email}) {
        $self->last_message($self->translate('The email is required.'));
        return 'error';
    }
    
    # Valid email format.
    unless($param->{email} =~ /^[^@]+@.+$/) {
        $self->last_message($self->translate('The email is invalid format.'));
        return 'error';
    }
    
    # Request url.
    my $url = $self->{register_url};
    unless($url) {
        $self->last_message($self->translate('The register_url is not defined.'));
        return 'error';
    }

    my ($result, $message);
    if ( !$self->{local_activation} ) {
        # Post the license.
        my $response = $self->post_license($url, {
            licenseKey      => $param->{licensekey},
            mailAddress     => $param->{email},
            hint            => $param->{hint},
        });
        
        # Connection error.
        if(!$response->is_success || !$response->headers->header('X-Result')) {
            $self->last_message($self->translate('Failure to connect to the licensing server for registration.'));
            return 'error';
        }

        # Get results..
        $result = $response->headers->header('X-Result');
        $message = $self->normalize_response($response->headers->header('X-Message'));
    } else {
        # Get local hash 
        require File::Spec;
        my $plugin = $self->{plugin};
        my $path = File::Spec->catfile($plugin->{full_path}, 'tmp', 'license.md5');
        unless ( -f $path ) {
            $result = 'error';
            $message = $self->translate('Cannot activate your license.');
        } else {
            open my $fh, "<$path";
            my $correct = '';
            while ( <$fh> ) {
                $correct .= $_;
            }
            close $fh;
            my $hash = md5_hex($param->{licensekey});
            if ( $correct && $correct == $hash ) {
                $result = 'success';
                $message = $self->translate('Thank you for your purchase. The license key has activated.');
            } else {
                $result = 'error';
                $message = $self->translate('Cannot activate your license.');
            }
        }
    } 
    # Check results.
    if($result ne 'error') {
        # Save the license data.
        $self->confirmation_message('');
        $self->licensekey($param->{licensekey});
        $self->email($param->{email});
        $self->save;
    }
    
    # Set complete message.
    $self->last_message($message);

    return $result;
}

# Confirm the license.
sub confirm_license {
    my $self = shift;
    my ($param) = @_;
    my $post = {};

    # Skip once
    return 'complete';
    
    # Request url.
    my $url = $self->{confirm_url} || return 'complete';
    
    # Reset the confirmation message.
    $self->confirmation_message('');
    
    # Post confirmation.
    my $response = $self->post_license($url, {
        licenseKey      => $self->licensekey,
    });
    
    # Update confirmation time.
    $self->confirmed_time(time);
    $self->save;
    
    # Connection error.
    if(!$response->is_success || !$response->headers->header('X-Result')) {
        MT::log($self->translate('Failure to connect to the licensing server for confirmation.'));
        return 'warning';
    }
    
    # Get results.
    my $result = $response->headers->header('X-Result');
    my $message = $self->normalize_response($response->headers->header('X-Message'));
    
    # Check results.
    if($result eq 'error') {
        $self->confirmation_message($message);
        
        # Reset the license.
        $self->licensekey('');
        $self->email('');
        $self->save;
    } elsif($result ne 'complete') {
        MT::log($message) if $message;
    }
    
    return $result;
}

1;